<?php
/**
 * User Profile API
 * 
 * This file handles API requests for user profile management
 * from the Android application.
 */

// Database connection parameters
$host = "localhost";
$db_name = "fncenphy_1xbetbd";
$username = "fncenphy_1xbetbd";
$password = "fncenphy_1xbetbd";
$conn = null;

// Set headers
header("Content-Type: application/json; charset=UTF-8");
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: GET, POST");
header("Access-Control-Allow-Headers: Content-Type, Access-Control-Allow-Headers, Authorization, X-Requested-With");

// Connect to database
function connect_db() {
    global $host, $db_name, $username, $password, $conn;
    
    try {
        $conn = new PDO("mysql:host=$host;dbname=$db_name", $username, $password);
        $conn->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        return true;
    } catch(PDOException $e) {
        send_response(false, "Database connection failed: " . $e->getMessage(), null);
        return false;
    }
}

// Function to send JSON response
function send_response($success, $message, $data = null) {
    $response = [
        "success" => $success,
        "message" => $message
    ];
    
    if ($data !== null) {
        $response["data"] = $data;
    }
    
    echo json_encode($response);
    exit;
}

// Get user data by account number
function get_user_data($account_number) {
    global $conn;
    
    try {
        // Prepare the SQL query
        $query = "SELECT 
                    id, account_number, email, phone, full_name, surname, patronymic,
                    country, city, account_created_datetime
                  FROM users 
                  WHERE account_number = :account_number
                  LIMIT 1";
        
        $stmt = $conn->prepare($query);
        $stmt->bindParam(":account_number", $account_number);
        $stmt->execute();
        
        // Check if user exists
        if ($stmt->rowCount() > 0) {
            $user = $stmt->fetch(PDO::FETCH_ASSOC);
            
            // Format the date for better display
            if (isset($user['account_created_datetime']) && $user['account_created_datetime']) {
                $date = new DateTime($user['account_created_datetime']);
                $user['formatted_date'] = $date->format('d/m/Y');
            } else {
                $user['formatted_date'] = date('d/m/Y'); // Use current date as fallback
            }
            
            send_response(true, "User data retrieved successfully", $user);
        } else {
            send_response(false, "User not found", null);
        }
    } catch(PDOException $e) {
        send_response(false, "Error retrieving user data: " . $e->getMessage(), null);
    }
}

// Update username
function update_username($account_number, $username) {
    global $conn;
    
    try {
        // Check if username already exists
        $check_query = "SELECT id FROM users WHERE full_name = :username AND account_number != :account_number";
        $check_stmt = $conn->prepare($check_query);
        $check_stmt->bindParam(":username", $username);
        $check_stmt->bindParam(":account_number", $account_number);
        $check_stmt->execute();
        
        if ($check_stmt->rowCount() > 0) {
            send_response(false, "Username already exists", null);
            return;
        }
        
        // Update the username
        $query = "UPDATE users SET full_name = :username WHERE account_number = :account_number";
        $stmt = $conn->prepare($query);
        $stmt->bindParam(":username", $username);
        $stmt->bindParam(":account_number", $account_number);
        
        if ($stmt->execute()) {
            send_response(true, "Username updated successfully", null);
        } else {
            send_response(false, "Failed to update username", null);
        }
    } catch(PDOException $e) {
        send_response(false, "Error updating username: " . $e->getMessage(), null);
    }
}

// Update email
function update_email($account_number, $email) {
    global $conn;
    
    try {
        // Validate email format
        if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            send_response(false, "Invalid email format", null);
            return;
        }
        
        // Check if email already exists
        $check_query = "SELECT id FROM users WHERE email = :email AND account_number != :account_number";
        $check_stmt = $conn->prepare($check_query);
        $check_stmt->bindParam(":email", $email);
        $check_stmt->bindParam(":account_number", $account_number);
        $check_stmt->execute();
        
        if ($check_stmt->rowCount() > 0) {
            send_response(false, "Email already in use", null);
            return;
        }
        
        // Update the email
        $query = "UPDATE users SET email = :email WHERE account_number = :account_number";
        $stmt = $conn->prepare($query);
        $stmt->bindParam(":email", $email);
        $stmt->bindParam(":account_number", $account_number);
        
        if ($stmt->execute()) {
            send_response(true, "Email updated successfully", null);
        } else {
            send_response(false, "Failed to update email", null);
        }
    } catch(PDOException $e) {
        send_response(false, "Error updating email: " . $e->getMessage(), null);
    }
}

// Update phone
function update_phone($account_number, $phone) {
    global $conn;
    
    try {
        // Check if phone already exists
        $check_query = "SELECT id FROM users WHERE phone = :phone AND account_number != :account_number";
        $check_stmt = $conn->prepare($check_query);
        $check_stmt->bindParam(":phone", $phone);
        $check_stmt->bindParam(":account_number", $account_number);
        $check_stmt->execute();
        
        if ($check_stmt->rowCount() > 0) {
            send_response(false, "Phone number already in use", null);
            return;
        }
        
        // Update the phone
        $query = "UPDATE users SET phone = :phone WHERE account_number = :account_number";
        $stmt = $conn->prepare($query);
        $stmt->bindParam(":phone", $phone);
        $stmt->bindParam(":account_number", $account_number);
        
        if ($stmt->execute()) {
            send_response(true, "Phone number updated successfully", null);
        } else {
            send_response(false, "Failed to update phone number", null);
        }
    } catch(PDOException $e) {
        send_response(false, "Error updating phone: " . $e->getMessage(), null);
    }
}

// Update password
function update_password($account_number, $old_password, $new_password) {
    global $conn;
    
    try {
        // Get current password hash
        $query = "SELECT password FROM users WHERE account_number = :account_number";
        $stmt = $conn->prepare($query);
        $stmt->bindParam(":account_number", $account_number);
        $stmt->execute();
        
        if ($stmt->rowCount() > 0) {
            $user = $stmt->fetch(PDO::FETCH_ASSOC);
            $current_hash = $user['password'];
            
            // Verify old password
            if (password_verify($old_password, $current_hash) || md5($old_password) === $current_hash) {
                // Hash new password
                $new_hash = password_hash($new_password, PASSWORD_DEFAULT);
                
                // Update the password
                $update_query = "UPDATE users SET password = :password WHERE account_number = :account_number";
                $update_stmt = $conn->prepare($update_query);
                $update_stmt->bindParam(":password", $new_hash);
                $update_stmt->bindParam(":account_number", $account_number);
                
                if ($update_stmt->execute()) {
                    send_response(true, "Password updated successfully", null);
                } else {
                    send_response(false, "Failed to update password", null);
                }
            } else {
                send_response(false, "Current password is incorrect", null);
            }
        } else {
            send_response(false, "User not found", null);
        }
    } catch(PDOException $e) {
        send_response(false, "Error updating password: " . $e->getMessage(), null);
    }
}

// Update personal information
function update_personal_info($account_number, $field, $value) {
    global $conn;
    
    // Define allowed fields to update
    $allowed_fields = [
        'name' => 'surname',
        'surname' => 'patronymic',
        'country' => 'country',
        'city' => 'city'
    ];
    
    // Check if field is allowed
    if (!isset($allowed_fields[$field])) {
        send_response(false, "Field not allowed to update", null);
        return;
    }
    
    $db_field = $allowed_fields[$field];
    
    try {
        // Update the personal info
        $query = "UPDATE users SET $db_field = :value WHERE account_number = :account_number";
        $stmt = $conn->prepare($query);
        $stmt->bindParam(":value", $value);
        $stmt->bindParam(":account_number", $account_number);
        
        if ($stmt->execute()) {
            send_response(true, "Personal information updated successfully", null);
        } else {
            send_response(false, "Failed to update personal information", null);
        }
    } catch(PDOException $e) {
        send_response(false, "Error updating personal information: " . $e->getMessage(), null);
    }
}

// Main logic to handle requests
if ($_SERVER['REQUEST_METHOD'] === 'GET') {
    if (isset($_GET['account_number'])) {
        $account_number = trim($_GET['account_number']);
        
        if (empty($account_number)) {
            send_response(false, "Account number is required", null);
        }
        
        if (connect_db()) {
            get_user_data($account_number);
        }
    } else {
        send_response(false, "Account number parameter is missing", null);
    }
} elseif ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Check for required fields
    if (!isset($_POST['account_number']) || !isset($_POST['action'])) {
        send_response(false, "Account number and action are required", null);
    }
    
    $account_number = trim($_POST['account_number']);
    $action = trim($_POST['action']);
    
    if (empty($account_number)) {
        send_response(false, "Account number is required", null);
    }
    
    if (connect_db()) {
        switch ($action) {
            case 'update_username':
                if (!isset($_POST['username'])) {
                    send_response(false, "Username is required", null);
                }
                $username = trim($_POST['username']);
                update_username($account_number, $username);
                break;
                
            case 'update_email':
                if (!isset($_POST['email'])) {
                    send_response(false, "Email is required", null);
                }
                $email = trim($_POST['email']);
                update_email($account_number, $email);
                break;
                
            case 'update_phone':
                if (!isset($_POST['phone'])) {
                    send_response(false, "Phone is required", null);
                }
                $phone = trim($_POST['phone']);
                update_phone($account_number, $phone);
                break;
                
            case 'update_password':
                if (!isset($_POST['old_password']) || !isset($_POST['new_password'])) {
                    send_response(false, "Old and new passwords are required", null);
                }
                $old_password = $_POST['old_password'];
                $new_password = $_POST['new_password'];
                update_password($account_number, $old_password, $new_password);
                break;
                
            case 'update_personal_info':
                if (isset($_POST['name'])) {
                    update_personal_info($account_number, 'name', trim($_POST['name']));
                } else if (isset($_POST['surname'])) {
                    update_personal_info($account_number, 'surname', trim($_POST['surname']));
                } else if (isset($_POST['country'])) {
                    update_personal_info($account_number, 'country', trim($_POST['country']));
                } else if (isset($_POST['city'])) {
                    update_personal_info($account_number, 'city', trim($_POST['city']));
                } else {
                    send_response(false, "No personal information field provided", null);
                }
                break;
                
            default:
                send_response(false, "Invalid action", null);
        }
    }
} else {
    // Method not allowed
    http_response_code(405);
    send_response(false, "Method not allowed", null);
}
?> 